import datetime
import re


class DiscourseConverter:
    def __init__(self, speaker, dt, venue=None, source_link=None):
        self.speaker = speaker
        self.dt = dt
        self.venue = venue
        self.source_link = source_link

        # start off needing first speaker
        self.need_speaker = speaker
        self.quote_state = []

    def need_add_quote_header(self):
        if not self.quote_state:
            return False

        if not self.quote_state[-1]['author']:
            return False

        if self.quote_state[-1]['added_header']:
            return False

        return True

    def process_p(self, p_content):
        # replace a \* with a {STARBIT}
        p_content = p_content.replace('\\*', '{STARBIT}')

        # if there is a "some words" then replace quotes with “some words” smart quotes
        p_content = re.sub(r'"([^"]+)"', r'“\1”', p_content)

        # replace a word with an apos't in it
        p_content = re.sub(r"(\w)'(\w)", r'\1’\2', p_content)

        # now for what is inside 'apostrophe'
        p_content = re.sub(r"'([^']+)'", r'‘\1’', p_content)

        # -- replace with nice hyphen
        p_content = p_content.replace('--', '–')

        # replace an isolated word of just "-" with nice hyphen
        p_content = re.sub(r' - ', ' – ', p_content)

        # ... replace with nice ellipsis
        p_content = p_content.replace('...', '…')

        # replace ***words words*** with <b><i>words words</i></b>
        p_content = re.sub(r'\*\*\*(.+?)\*\*\*', r'<b><i>\1</i></b>', p_content)

        # similar for *words* italic and **words** bold
        p_content = re.sub(r'\*\*(.+?)\*\*', r'<b>\1</b>', p_content)
        p_content = re.sub(r'\*(.+?)\*', r'<i>\1</i>', p_content)

        # _words words_ is italic
        p_content = re.sub(r'_([^_]+?)_', r'<i>\1</i>', p_content)

        # add back an actual * and * around <b> and </b>
        p_content = p_content.replace('<b>', '<b>*').replace('</b>', '*</b>')

        # replace № with encoding
        p_content = p_content.replace('№', '&#x2116;')

        # ----------------------------
        # if have numbers followed by st, nd, rd, th then replace with <sup>st</sup> etc
        p_content = re.sub(r'(\d+)(st|nd|rd|th)', r'\1<sup>\2</sup>', p_content)

        # --------------------------

        # for a [[source](http://www.actualfreedom.com.au/directroute/05.htm#17Jan10)]
        # replace like         <span class="source-link">
        #           <a href="href" target="_blank"></a>
        #           [TODO] FILL HERE]
        #         </span>
        p_content = re.sub(r'\[\[source\]\((.+?)\)\]', r'''
<span class="source-link">
    <a href="\1" target="_blank"></a>
    [TODO FILL HERE]
</span>''', p_content)

        # same for link
        p_content = re.sub(r'\[\[link\]\((.+?)\)\]', r'''
<span class="source-link">
    <a href="\1" target="_blank"></a>
    [TODO FILL HERE]
</span>''', p_content)

        # for any other kind of [words words](http://urlhere) then:
        # if url is to 'actualfreedom':
        #     Inner link<a class="footnote-link" href="http://an.actualfreedom.com.au/various/well-equippedmarvelling.htm" target="_blank"></a>,
        # otherwise:
        #     external link<a class="footnote-link external" href="http://discuss.actualism.online" target="_blank"></a>,
        p_content = re.sub(r'\[([^\]]+?)\]\((.+?)actualfreedom.com.au(.+?)\)', r'''
\1<a class="footnote-link" href="\2actualfreedom.com.au\3" target="_blank"></a>''', p_content)

        p_content = re.sub(r'\[([^\]]+?)\]\((.+?)\)', r'''
\1<a class="footnote-link external" href="\2" target="_blank"></a>''', p_content)

        # replace any :words: with <emoji class="words"></emoji>"
        p_content = re.sub(r':([\w_-]+):', r'<emoji class="\1"></emoji>', p_content)

        # 😁 --> <emoji class="grin"></emoji>
        p_content = re.sub(r'😁', r'<emoji class="grin"></emoji>', p_content)

        # 😄 --> <emoji class="smile"></emoji>
        p_content = re.sub(r'😄', r'<emoji class="smile"></emoji>', p_content)

        # 😝 --> <emoji class="stuck-out-tongue-closed-eyes"></emoji>
        p_content = re.sub(r'😝', r'<emoji class="stuck-out-tongue-closed-eyes"></emoji>', p_content)

        # 🤔 --> <emoji class="thinking"></emoji>
        p_content = re.sub(r'🤔', r'<emoji class="thinking"></emoji>', p_content)

        # 😆 --> <emoji class="laughing"></emoji>
        p_content = re.sub(r'😆', r'<emoji class="laughing"></emoji>', p_content)

        # 😊 --> <emoji class="blush"></emoji>
        p_content = re.sub(r'😊', r'<emoji class="blush"></emoji>', p_content)

        # ☺️ --> <emoji class="blush"></emoji>
        p_content = re.sub(r'☺️', r'<emoji class="blush"></emoji>', p_content)

        # INLINE QUOTES --
        # [ma]some words[/ma] --> <span class="quote mainauth">some words</span>
        p_content = re.sub(r'\[ma\](.+?)\[/ma\]', r'<span class="quote mainauth">\1</span>', p_content)
        # [c]some words[/c] --> <span class="quote claudiu">some words</span>
        p_content = re.sub(r'\[c\](.+?)\[/c\]', r'<span class="quote claudiu">\1</span>', p_content)
        # same for [r][/r] and richard
        p_content = re.sub(r'\[r\](.+?)\[/r\]', r'<span class="quote richard">\1</span>', p_content)
        # and [v] with vineeto
        p_content = re.sub(r'\[v\](.+?)\[/v\]', r'<span class="quote vineeto">\1</span>', p_content)
        # and [q] for a plain quote
        p_content = re.sub(r'\[q\](.+?)\[/q\]', r'<span class="quote">\1</span>', p_content)

        # -------------------------------
        username_mappings = {
            'JonnyPitt': 'Jonathan',
            "Kub933": 'Kuba',
            'jamesjjoo': 'James',
        }

        # map @username to its mapping, and capitalize the first letter and lowercase the rest, remove @ as well
        p_content = re.sub(r'@(\w+)', lambda m: username_mappings.get(m.group(1), m.group(1).capitalize()), p_content)

        # -------------------------------
        # replace {STARBIT} back with a *
        p_content = p_content.replace('{STARBIT}', '*')

        return p_content

    def handle_line(self, line):
        if not line.strip():
            return []

        # if not in a quote...
        if not self.quote_state:
            # if line.strip() only has - in it then replace with fancy line separator
            if all(c == '-' for c in line.strip()):
                return ['<hr class="inner-entry">']

            # same for '*'
            if all(c == '*' for c in line.strip()):
                return ['<hr class="inner-entry">']

        if '[quote' in line:
            indent_class = 'quote-indent' if 'indent' in line else ''

            # after quote ends we will need the speaker
            if '[quote]' in line:
                self.need_speaker = None
                self.quote_state.append({'author': None, 'first': True, 'added_header': False})
                return [f'<div class="quote {indent_class}">']

            # extract the author
            author = line.split('="')[1].split('"')[0].split(',')[0].strip().lower()

            self.need_speaker = None
            self.quote_state.append({'author': author, 'first': True, 'added_header': False})

            authclass = author
            if authclass == self.speaker:
                authclass = "mainauth"
            return [f'''<div class="quote {authclass} {indent_class}">''']

        if '[/quote]' in line:
            qs = self.quote_state.pop()
            if self.quote_state:
                self.need_speaker = self.quote_state[-1]['author']
            else:
                self.need_speaker = self.speaker
            return ['</div>']

        res_bits = ['<p>']

        if self.need_add_quote_header():
            self.need_speaker = None
            res_bits.append(f'''<span class="quote-header">{self.quote_state[-1]['author']}:</span>''')
            self.quote_state[-1]['added_header'] = True

        if self.need_speaker:
            res_bits.append(f'''<span class="speaker">{self.need_speaker}:</span>''')
            self.need_speaker = None

        res_bits.append(self.process_p(line))

        res_bits.append('</p>')

        return res_bits

    def run(self, data):
        lines = data.split('\n')

        output = []

        for line in lines:
            output.extend(self.handle_line(line))

        # turn date to id bit like 25Jun24
        id_bit = self.dt.strftime('%d%b%y')

        # make date header out of dt like June 25, 2024, 2:16 PM WEST
        date_header = self.dt.strftime('%B %d, %Y, %-I:%M %p %Z')

        # date end line like 2 June 2024
        date_end = self.dt.strftime('%-d %B %Y')

        # add indent level to lines
        indent_level = 2
        for i, line in enumerate(output):
            if '</div>' in line or '</p>' in line:
                indent_level -= 1

            output[i] = '  ' * indent_level + line

            if '<div' in line or '<p' in line:
                indent_level += 1


        return f"""\
  <hr class="entry-sep">

  <div class="entry mainauth">
    <a class="date-header" name="{id_bit}" href="#{id_bit}">{date_header}</a>
%s
    <p>
      <span class="source-link">
        {"<a href=" + self.source_link + ' target="_blank"></a>' if self.source_link else ''} 
        {self.speaker.capitalize()}, {self.venue}, {date_end}
      </span>
    </p>
  </div>
""" % '\n'.join(output)













data = """
Since yesterday I have been fascinated by the fact that allowing happiness and harmlessness **is** the most caring and selfless thing to be done. It is interesting because within 'humanity' it is held that it is the one who is prepared to suffer the most who is a good person, that the one who allows happiness for themselves is selfish or has sold their soul to the devil, yet the facts are exactly the opposite. 

Firstly it is impossible to be genuinely happy without at the same time being harmless, those people who are seen to cause chaos in pursuit of 'happiness' are actually deeply unhappy and thus desperately looking for the next fix of good feelings. 

Also it is those people who are devoured by their various dramas and demons who are the most self involved, and in the process they are the ones causing the most harm both to themselves and others. Now they have a choice - to be happy (and therefore harmless) a choice that would benefit both them and those around them (everybody wins) and yet this choice would require that they relinquish a precious part of 'themselves'. 

The choice to remain as 'I' am and thus continue causing harm to all is the uncaring one, it is the selfish one. The choice to allow happiness and harmlessness and thus benefit all is the caring one, and it is the selfless one. 

I remember reading Peter mention this, that merely chasing after 'my' gratification is insufficient motivation, and this makes sense now. Because if 'I' am motivated in this self-centred way 'I' will remain exactly as 'I' am and thus continue causing harm to all, neither happiness nor harmlessness will be allowed as 'I' will choose to remain unchanged. 

They key then is to see the full picture, to see the harm 'I' am causing to **all concerned** (including myself) by remaining as 'I' am. I find it fascinating how this segues into actual freedom, that in the end to ensure actual (and irrevocable) happiness and harmlessness (and thus to benefit all) 'I' have to give up 'myself' altogether.
"""

import pytz
# print(DiscourseConverter(
#     "claudiu",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 9, 20, 15, 2)),
#     venue='DAO: Claudiu’s Journal',
#     source_link='https://discuss.actualism.online/t/claudius-journal/274/228',
# ).run(data))
# print(DiscourseConverter(
#     "claudiu",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 8, 24, 13, 51)),
#     venue='E-mail to Vineeto',
#     source_link=None,
# ).run(data))
# print(DiscourseConverter(
#     "vineeto",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 8, 21, 9, 57)),
#     venue='E-mail to Claudiu',
#     source_link=None,  # 'https://discuss.actualism.online/t/claudius-journal/274/181',
# ).run(data))
# print(DiscourseConverter(
#     "claudiu",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 9, 13, 12, 00)),
#     venue='DAO: Reports on being Out-from-Control',
#     source_link='https://discuss.actualism.online/t/reports-on-being-out-from-control/1018/6',
# ).run(data))

# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 8, 29, 16, 2)),
#     venue='DAO: Claudiu’s Journal',
#     source_link='https://discuss.actualism.online/t/claudius-journal/274/224',
# ).run(data))
print(DiscourseConverter(
    "kuba",
    dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 9, 16, 11, 17)).astimezone(pytz.timezone('Europe/London')),
    venue='DAO: Kub933’s Journal',
    source_link='https://discuss.actualism.online/t/kub933s-journal/467/1122',
).run(data))
# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 7, 26, 21, 33)).astimezone(pytz.timezone('Europe/London')),
#     venue='DAO: James’ Journal',
#     source_link='https://discuss.actualism.online/t/james-journal/272/411',
# ).run(data))
# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 7, 15, 12, 00)),
#     venue='DAO: Torch bearer',
#     source_link='https://discuss.actualism.online/t/torch-bearer/1005/7',
# ).run(data))
# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 7, 21, 18, 51)),
#     venue='DAO: This moment has no duration',
#     source_link='https://discuss.actualism.online/t/this-moment-has-no-duration/286/119',
# ).run(data))
# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 8, 5, 23, 13)).astimezone(pytz.timezone('Europe/London')),
#     venue='DAO: Felipe’s reflections & heuristics',
#     source_link='https://discuss.actualism.online/t/felipes-reflections-heuristics/896/7',
# ).run(data))
# print(DiscourseConverter(
#     "kuba",
#     dt=pytz.timezone('Europe/Lisbon').localize(datetime.datetime(2024, 9, 13, 12, 18)).astimezone(pytz.timezone('Europe/London')),
#     venue='DAO: Reports on being Out-from-Control',
#     source_link='https://discuss.actualism.online/t/reports-on-being-out-from-control/1018/7',
# ).run(data))